
bool rm_rf(const String &path);
extern bool systemFolder;
extern bool coreFolder;

class TarExtractor {
public:
    TarExtractor() {
        reset();
    }

    void begin(const String &tarFileName) {
        reset();

        enabled = tarFileName.endsWith(".tar");
        if (!enabled) return;

        // fullFS.tar → rootba csomagolunk
        if (tarFileName == "fullFS.tar") {
            baseDir = "";
            return;
        }

        baseDir = "/" + tarFileName;
        int dot = baseDir.lastIndexOf('.');
        if (dot > 0) baseDir = baseDir.substring(0, dot);

        if (!LittleFS.exists(baseDir)) {
            LittleFS.mkdir(baseDir);
        }
    }

    bool isEnabled() const { return enabled; }

    void processChunk(const uint8_t *data, size_t len) {
        if (!enabled) return;

        size_t offset = 0;

        while (offset < len) {
            size_t toCopy = min(bytesNeeded, len - offset);

            switch (state) {

            case READ_HEADER:
                memcpy(header + headerPos, data + offset, toCopy);
                headerPos += toCopy;
                bytesNeeded -= toCopy;

                if (bytesNeeded == 0) {
                    if (isEndOfArchive()) {
                        reset();
                        return;
                    }

                    parseHeader();

                    // ha könyvtár volt, parseHeader már elintézte, lépünk tovább
                    if (fileName.length() == 0 && fileSize == 0) {
                        state = READ_HEADER;
                        bytesNeeded = 512;
                        headerPos = 0;
                        break;
                    }

                    if (fileSize > 0 && fileName.length() > 0) {
                        openOutputFile();
                        state = READ_FILEDATA;
                        bytesNeeded = fileSize;
                    } else {
                        state = SKIP_PADDING;
                        bytesNeeded = paddingFor(0);
                    }

                    headerPos = 0;
                }
                break;

            case READ_FILEDATA:
                if (currentFile) currentFile.write(data + offset, toCopy);
                bytesNeeded -= toCopy;

                if (bytesNeeded == 0) {
                    currentFile.close();
                    state = SKIP_PADDING;
                    bytesNeeded = paddingFor(fileSize);
                }
                break;

            case SKIP_PADDING:
                bytesNeeded -= toCopy;
                if (bytesNeeded == 0) {
                    state = READ_HEADER;
                    bytesNeeded = 512;
                }
                break;
            }

            offset += toCopy;
        }
    }

private:
    bool enabled = false;

    enum State { READ_HEADER, READ_FILEDATA, SKIP_PADDING };
    State state;

    uint8_t header[512];
    size_t headerPos;

    size_t bytesNeeded;
    size_t fileSize;

    String baseDir;
    String fileName;
    File currentFile;

    void reset() {
        enabled = false;
        state = READ_HEADER;
        headerPos = 0;
        bytesNeeded = 512;
        fileSize = 0;
        fileName = "";
        currentFile.close();
    }

    bool isEndOfArchive() {
        for (int i = 0; i < 512; i++) {
            if (header[i] != 0) return false;
        }
        return true;
    }

    size_t paddingFor(size_t size) {
        return (512 - (size % 512)) % 512;
    }

    void parseHeader() {
        char name[101];
        memcpy(name, header, 100);
        name[100] = 0;
        fileName = String(name);

        char sizeStr[13];
        memcpy(sizeStr, header + 124, 12);
        sizeStr[12] = 0;
        fileSize = strtol(sizeStr, nullptr, 8);

        char typeflag = header[156];
        bool isDir = (typeflag == '5') || fileName.endsWith("/");

        if (isDir) {
            String full = baseDir + "/" + fileName;
            if (full.endsWith("/")) full.remove(full.length() - 1);
            if (full.length() == 0) full = "/";

            if (!LittleFS.exists(full)) {
                LittleFS.mkdir(full);
            }

            // jelöljük, hogy ez könyvtár volt, nincs filedata
            fileName = "";
            fileSize = 0;
            return;
        }

        if (fileName.endsWith("/")) fileName = "";
    }

    void openOutputFile() {
        String full = baseDir + "/" + fileName;

        if (!systemFolder && full.startsWith("/system/")) {
            Serial.println("TAR: system folder write blocked");
            return;
        }
        if (!coreFolder && full.startsWith("/core/")) {
            Serial.println("TAR: core folder write blocked");
            return;
        }

        int slash = full.lastIndexOf('/');
        if (slash > 0) {
            String dir = full.substring(0, slash);
            if (!LittleFS.exists(dir)) LittleFS.mkdir(dir);
        }

        currentFile = LittleFS.open(full, "w");
    }
};



class TarResponse : public AsyncAbstractResponse {
public:
    TarResponse(const String &rootFsPath)
        : rootFs(rootFsPath)
    {
        _code = 200;
        _sendContentLength = false;
        _chunked = true;
        _contentLength = 0;

        setContentType("application/x-tar");

        // Root mappa neve
        String rootName = rootFsPath;
        if (rootName.endsWith("/")) rootName.remove(rootName.length() - 1);
        rootName = rootName.substring(rootName.lastIndexOf('/') + 1);

        // ===== ROOT FILTER =====
        if ((!systemFolder && rootName == "system") ||
            (!coreFolder   && rootName == "core"))
        {
            state = DONE;
            eofBlocks = 2;
            return;
        }

        stack.push_back({rootFs, ""});

        state = SEND_HEADER;
        eofBlocks = 0;
    }

    bool _sourceValid() const override {
        return true;
    }

    size_t _fillBuffer(uint8_t *buf, size_t maxLen) override {
        size_t outLen = 0;

        while (outLen == 0) {

            if (state == DONE) {
                return 0;
            }

            if (state == SEND_HEADER) {
                if (makeNextHeader(buf)) {
                    outLen = 512;
                    state = currentIsDir ? NEXT_ENTRY : SEND_DATA;
                    return outLen;
                } else {
                    state = SEND_EOF;
                }
            }

            if (state == SEND_DATA) {
                if (sendFileData(buf, maxLen, outLen)) {
                    return outLen;
                } else {
                    state = NEXT_ENTRY;
                }
            }

            if (state == NEXT_ENTRY) {
                if (!hasMoreEntries()) {
                    state = SEND_EOF;
                } else {
                    state = SEND_HEADER;
                }
            }

            if (state == SEND_EOF) {
                if (eofBlocks < 2) {
                    memset(buf, 0, 512);
                    eofBlocks++;
                    return 512;
                }
                state = DONE;
                return 0;
            }
        }

        return outLen;
    }

private:
    enum State {
        SEND_HEADER,
        SEND_DATA,
        NEXT_ENTRY,
        SEND_EOF,
        DONE
    };

    struct Entry {
        String fsPath;
        String tarPath;
    };

    State state;
    String rootFs;
    std::vector<Entry> stack;

    File currentFile;
    size_t fileBytesSent = 0;
    bool currentIsDir = false;
    int eofBlocks = 0;

    // ---- TAR header készítés ----
    void makeTarHeader(uint8_t *block,
                       const String &name,
                       size_t size,
                       bool isDir)
    {
        memset(block, 0, 512);

        snprintf((char*)block +   0, 100, "%s", name.c_str());
        snprintf((char*)block + 100,   8, "%07o", isDir ? 0755 : 0644);
        snprintf((char*)block + 108,   8, "%07o", 0);
        snprintf((char*)block + 116,   8, "%07o", 0);
        snprintf((char*)block + 124,  12, "%011o", size);
        snprintf((char*)block + 136,  12, "%011o", (unsigned int)time(nullptr));

        memset(block + 148, ' ', 8);

        block[156] = isDir ? '5' : '0';

        memcpy(block + 257, "ustar", 5);
        block[262] = '\0';
        memcpy(block + 263, "00", 2);

        unsigned int sum = 0;
        for (int i = 0; i < 512; i++) sum += block[i];

        snprintf((char*)block + 148, 8, "%06o", sum);
        block[154] = '\0';
        block[155] = ' ';
    }

    // ---- Következő header ----
    bool makeNextHeader(uint8_t *block) {
        if (stack.empty()) return false;

        Entry e = stack.back();
        stack.pop_back();

        File f = LittleFS.open(e.fsPath);
        if (!f) return false;

        currentIsDir = f.isDirectory();
        size_t size = currentIsDir ? 0 : f.size();

        String tarName = e.tarPath;
        if (currentIsDir && !tarName.endsWith("/")) tarName += "/";

        makeTarHeader(block, tarName, size, currentIsDir);

        if (currentIsDir) {
            File c;
            while ((c = f.openNextFile())) {
                String fullName = String(c.name());
                String baseName = fullName.substring(fullName.lastIndexOf('/') + 1);

                // ===== SYSTEM FOLDER FILTER =====
                if (!systemFolder && baseName == "system") {
                    c.close();
                    continue;
                }

                // ===== CORE FOLDER FILTER =====
                if (!coreFolder && baseName == "core") {
                    c.close();
                    continue;
                }

                String childFs = e.fsPath;
                if (!childFs.endsWith("/")) childFs += "/";
                childFs += baseName;

                String childTar = tarName + baseName;

                stack.push_back({childFs, childTar});
                c.close();
            }
            f.close();
            currentFile = File();
            fileBytesSent = 0;
        } else {
            currentFile = f;
            fileBytesSent = 0;
        }

        return true;
    }

    // ---- Fájl adat küldése ----
    bool sendFileData(uint8_t *buffer, size_t maxLen, size_t &outLen) {
        outLen = 0;
        if (!currentFile) return false;

        size_t toRead = maxLen;
        if (toRead > 4096) toRead = 4096;

        size_t n = currentFile.read(buffer, toRead);
        if (n > 0) {
            outLen = n;
            fileBytesSent += n;
            return true;
        }

        size_t pad = (512 - (fileBytesSent % 512)) % 512;
        if (pad) {
            memset(buffer, 0, pad);
            outLen = pad;
            fileBytesSent += pad;
            currentFile.close();
            currentFile = File();
            return true;
        }

        currentFile.close();
        currentFile = File();
        return false;
    }

    bool hasMoreEntries() {
        return !stack.empty();
    }
};


size_t dirSize(File dir){
    if(!dir || !dir.isDirectory()) return 0;
    size_t total = 0;
    File f = dir.openNextFile();
    while(f){
        if(f.isDirectory()){
            total += dirSize(f); // rekurzió File objektummal
        } else {
            total += f.size();
        }
        f.close();
        f = dir.openNextFile();
    }
    return total;
}


const char raw_upload_html[] PROGMEM = R"rawliteral(

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Flash File Manager</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
body {
  font-family: system-ui, sans-serif;
  background: #f5f6f8;
  margin: 0;
  padding: 20px;
  color: #333;
}

.card {
  background: #fff;
  border-radius: 8px;
  box-shadow: 0 2px 6px rgba(0,0,0,0.05);
  padding: 20px;
  margin-bottom: 20px;
  max-width: 600px;
}

h3 {
  margin-top: 0;
  font-size: 1.2em;
  color: #222;
}

#path {
  font-weight: bold;
  color: #666;
}

ul {
  list-style: none;
  padding: 0;
  margin: 10px 0;
  border: 1px solid #ddd;
  border-radius: 6px;
  background: #fff;
  max-height: 200px;
  overflow-y: auto;
}

li {
  padding: 8px 12px;
  border-bottom: 1px solid #eee;
  cursor: pointer;
  display: flex;
  justify-content: space-between;
  align-items: center;
}

li:last-child {
  border-bottom: none;
}

li.sel {
  background: #e0f0ff;
}

li.up {
  font-weight: bold;
}

.name {
  flex: 1;
}

.size {
  font-family: monospace;
  color: #888;
  white-space: nowrap;
}

button {
  background: #007bff;
  color: white;
  border: none;
  padding: 8px 14px;
  margin: 5px 4px 0 0;
  border-radius: 6px;
  cursor: pointer;
  font-size: 0.9em;
}

button.red {
  background: #c00;
}

button.disabled {
  background: #ccc;
  cursor: default;
}

#flash-status {
  height: 20px;
  background: #eee;
  border-radius: 6px;
  overflow: hidden;
  margin-bottom: 10px;
  border: 1px solid #ccc;
}

#flash-bar {
  height: 100%;
  background: #007bff;
  width: 0%;
}

#flash-info {
  font-size: 0.9em;
  color: #555;
  margin-bottom: 8px;
}

#drop {
  border: 2px dashed #bbb;
  border-radius: 6px;
  padding: 20px;
  text-align: center;
  color: #888;
  margin-bottom: 10px;
}
</style>
</head>
<body>

<div class="card">
  <h3>Flash Files</h3>
  <div>Path: <span id="path">/</span></div>
  <ul id="files"></ul>
  <div>
    <button onclick="mkdir()">MKDIR</button>
    <button id="rn" class="disabled" onclick="ren()">Rename</button>
    <button id="rm" class="disabled" onclick="del()">Delete</button>
    <button id="dl" class="disabled" onclick="download()">Download</button>
    <button class="red" onclick="formatFS()">FORMAT</button>
  </div>
</div>

<div class="card">
  <h3>Upload</h3>

  <div id="flash-info">Used: 0 kB / 0 kB</div>

  <div id="flash-status"><div id="flash-bar"></div></div>

  <div id="drop">Drag & Drop files here</div>

  <input type="file" id="f" multiple>
  <button onclick="uploadBtn()">Upload</button>
</div>

<form id="dlForm" method="POST" action="/download" target="_blank">
  <input type="hidden" name="path" id="dlPath">
</form>

<script>
let cwd="/", sel=null, selIsDir=false, selIsSystem=false;

function isProtectedRootFolder(name) {
  return (cwd === "/" && (name === "core" || name === "system"));
}

function fmtSize(sz){
  if(sz<1024) return sz+" B";
  return (sz/1024).toFixed(1)+" kB";
}

function btn(enable,dir){
  const rn = document.getElementById('rn');
  const rm = document.getElementById('rm');
  const dl = document.getElementById('dl');

  rn.classList.toggle('disabled', !enable);
  rm.classList.toggle('disabled', !enable);
  dl.classList.toggle('disabled', !enable);

  if (selIsSystem) {
    rn.classList.add('disabled');
    rm.classList.add('disabled');
  }
}

function renderEntry(f,type){
  let ul=document.getElementById('files');
  let li=document.createElement('li');
  let icon=f.dir?'📁 ':'📄 ';
  if(type==="system") icon='⚙️ ';
  if(type==="core") icon='🔺 ';
  let name=document.createElement('span');
  name.className='name';
  name.textContent=icon+f.name;
  let size=document.createElement('span');
  size.className='size';
  size.textContent=f.dir?"":fmtSize(f.size);
  li.appendChild(name);
  li.appendChild(size);

  const filePath = cwd + (cwd.endsWith('/')?'':'/') + f.name;

  li.onclick=()=>{
    [...ul.children].forEach(x=>x.classList.remove('sel'));
    li.classList.add('sel');

    sel = filePath;
    selIsDir = f.dir;

    selIsSystem = isProtectedRootFolder(f.name);

    btn(true, f.dir);
  };

  li.ondblclick=e=>{
    e.preventDefault();
    if(f.dir){
      cwd=filePath;
      load();
    }
  };

  ul.appendChild(li);
}

function load(){
  document.getElementById('path').textContent=cwd;
  loadFlashStatus();

  fetch('/list',{
    method:'POST',
    headers:{'Content-Type':'application/x-www-form-urlencoded'},
    body:'path='+encodeURIComponent(cwd)
  })
  .then(r=>r.json())
  .then(j=>{
    let ul=document.getElementById('files');
    ul.innerHTML=''; sel=null; selIsSystem=false; btn(false,false);

    let li=document.createElement('li');
    li.textContent='📁 ..';
    li.className='up';

    if(cwd !== "/") {
      li.ondblclick = () => {
        cwd = cwd.substring(0, cwd.lastIndexOf('/')) || "/";
        load();
      };
    } else {
      li.onclick = () => {
        [...ul.children].forEach(x=>x.classList.remove('sel'));
        li.classList.add('sel');
        sel = "/";
        selIsDir = true;
        selIsSystem = false;
        btn(false,false);
        document.getElementById('dl').classList.remove('disabled');
      };
    }

    ul.appendChild(li);

    let core=null,system=null,others=[];
    j.forEach(f=>{
      if(cwd==="/"&&f.dir&&f.name==="core") core=f;
      else if(cwd==="/"&&f.dir&&f.name==="system") system=f;
      else others.push(f);
    });

    if(core)renderEntry(core,"core");
    if(system)renderEntry(system,"system");
    others.forEach(f=>renderEntry(f,"normal"));
  });
}

function loadFlashStatus(){
  fetch('/flashfree').then(r=>r.json()).then(j=>{
    let bar=document.getElementById('flash-bar');
    let info=document.getElementById('flash-info');

    let percent=(j.used/j.total)*100;
    bar.style.width=percent+'%';

    info.textContent = "Used: " + j.used + " kB / " + j.total + " kB";
  });
}

function checkUploadSize(files,ok){
  let total=0;
  for(let i=0;i<files.length;i++) total+=files[i].size;
  fetch('/flashfree').then(r=>r.json()).then(j=>{
    let free=(j.total-j.used)*1024;
    if(total>free){ alert("Not enough flash space"); return; }
    ok();
  });
}

function uploadFiles(files){
  let i=0;
  let timer=setInterval(loadFlashStatus,300);
  (function next(){
    if(i>=files.length){ clearInterval(timer); load(); return; }
    upload(files[i++],next);
  })();
}

function upload(file,done){
  let x=new XMLHttpRequest();
  x.onload=()=>done&&done();
  x.open('POST','/upload?path='+encodeURIComponent(cwd)+'&name='+encodeURIComponent(file.name));
  x.send(file);
}

function uploadBtn(){
  let files=document.getElementById('f').files;
  if(files.length) checkUploadSize(files,()=>uploadFiles(files));
}

const drop = document.getElementById('drop');
drop.ondragover=e=>e.preventDefault();
drop.ondrop=e=>{
  e.preventDefault();
  let files=e.dataTransfer.files;
  if(files.length) checkUploadSize(files,()=>uploadFiles(files));
};

function mkdir(){
  let n=prompt("Dir name:"); if(!n)return;
  fetch('/mkdir',{
    method:'POST',
    headers:{'Content-Type':'application/x-www-form-urlencoded'},
    body:'path='+encodeURIComponent(cwd+"/"+n)
  }).then(load);
}

function del(){
  if(selIsSystem || !sel || !confirm("Delete?")) return;

  fetch('/delete',{
    method:'POST',
    headers:{'Content-Type':'application/x-www-form-urlencoded'},
    body:'path='+encodeURIComponent(sel)
  }).then(load);
}

function ren(){
  if(selIsSystem || !sel) return;

  let base = sel.substring(0, sel.lastIndexOf('/'));
  let n = prompt("New name:", sel.split('/').pop());
  if (!n) return;

  fetch('/rename',{
    method:'POST',
    headers:{'Content-Type':'application/x-www-form-urlencoded'},
    body:'old='+encodeURIComponent(sel)+'&new='+encodeURIComponent(base+'/'+n)
  }).then(load);
}

function download() {
  if (!sel) return;
  document.getElementById("dlPath").value = sel;
  document.getElementById("dlForm").submit();
}

function formatFS(){
  if(!confirm("Format LittleFS? All data will be lost!")) return;

  fetch('/format',{
    method:'POST',
    headers:{'Content-Type':'application/x-www-form-urlencoded'},
    body:'confirm=YES'
  }).then(load);
}

load();
</script>

</body>
</html>


)rawliteral";





/* ===================== /list ===================== */
struct Entry { String name; bool dir; size_t size; };

inline void handleList(AsyncWebServerRequest *r) {
    String path = "/";
    if (r->hasParam("path", true)) path = r->getParam("path", true)->value();

    File dir = LittleFS.open(path);
    if (!dir || !dir.isDirectory()) {
        r->send(200, "application/json", "[]");
        return;
    }

    std::vector<Entry> v;
    File x = dir.openNextFile();
    while (x) {
        String fullName = String(x.name());
        String n = fullName.substring(fullName.lastIndexOf('/') + 1);

        if (n == "..") { x = dir.openNextFile(); continue; }

        if (path == "/") {
            if (!coreFolder && n == "core") { x = dir.openNextFile(); continue; }
            if (!systemFolder && n == "system") { x = dir.openNextFile(); continue; }
        }

        size_t sz = x.isDirectory() ? dirSize(x) : x.size();
        v.push_back({n, x.isDirectory(), sz});
        x = dir.openNextFile();
    }

    std::sort(v.begin(), v.end(), [](const Entry &a, const Entry &b) {
        if (a.name.equalsIgnoreCase("core")) return true;
        if (b.name.equalsIgnoreCase("core")) return false;
        if (a.name.equalsIgnoreCase("system")) return true;
        if (b.name.equalsIgnoreCase("system")) return false;
        if (a.dir != b.dir) return a.dir;
        String A = a.name; A.toLowerCase();
        String B = b.name; B.toLowerCase();
        return A < B;
    });

    String j = "[";
    for (size_t i = 0; i < v.size(); i++) {
        if (i) j += ",";
        j += "{\"name\":\"" + v[i].name +
             "\",\"dir\":" + (v[i].dir ? "true" : "false") +
             ",\"size\":" + String(v[i].size) + "}";
    }
    j += "]";

    r->send(200, "application/json", j);
}

/* ===================== /list_s ===================== */
inline void handleListSafe(AsyncWebServerRequest *r) {
    String path = "/";
    if (r->hasParam("path", true))
        path = r->getParam("path", true)->value();

    File dir = LittleFS.open(path);
    if (!dir || !dir.isDirectory()) {
        r->send(200, "application/json", "[]");
        return;
    }

    std::vector<Entry> v;
    while (true) {
        File f = dir.openNextFile();
        if (!f) break;

        const char *raw = f.name();
        if (!raw) { f.close(); continue; }

        bool valid = true;
        for (int i = 0; i < 128; i++) {
            char c = raw[i];
            if (c == 0) break;
            if (c < 32 || c > 126) { valid = false; break; }
        }
        if (!valid) { f.close(); continue; }

        String full = String(raw);
        int idx = full.lastIndexOf('/');
        if (idx < 0) idx = -1;
        String name = full.substring(idx + 1);
        if (name.length() == 0) { f.close(); continue; }

        bool isDir = f.isDirectory();
        size_t sz = isDir ? 0 : f.size();

        v.push_back({name, isDir, sz});
        f.close();
    }

    std::sort(v.begin(), v.end(), [](const Entry &a, const Entry &b) {
        if (a.dir != b.dir) return a.dir;
        String A = a.name; A.toLowerCase();
        String B = b.name; B.toLowerCase();
        return A < B;
    });

    String j = "[";
    for (size_t i = 0; i < v.size(); i++) {
        if (i) j += ",";
        j += "{\"name\":\"" + v[i].name +
             "\",\"dir\":" + (v[i].dir ? "true" : "false") +
             ",\"size\":" + String(v[i].size) + "}";
    }
    j += "]";

    r->send(200, "application/json", j);
}

/* ===================== /flashfree ===================== */
inline void handleFlashFree(AsyncWebServerRequest *r) {
    r->send(200, "application/json",
        "{\"total\":" + String(LittleFS.totalBytes()/1024) +
        ",\"used\":" + String(LittleFS.usedBytes()/1024) + "}");
}

/* ===================== /upload ===================== */
inline void handleUpload(AsyncWebServerRequest *r, uint8_t *d, size_t l, size_t i, size_t t) {

    if (i == 0) {
        String name = r->getParam("name")->value();
        String path = r->getParam("path")->value();
        if (!path.endsWith("/")) path += "/";

        TarExtractor *tar = new TarExtractor();
        tar->begin(name);
        r->_tempObject = tar;

        if (!name.endsWith(".tar")) {
            r->_tempFile = LittleFS.open(path + name, "w");
        }
    }

    TarExtractor *tar = (TarExtractor*) r->_tempObject;

    if (tar && tar->isEnabled()) {
        tar->processChunk(d, l);
    } else {
        if (r->_tempFile) r->_tempFile.write(d, l);
    }

    if (i + l == t) {
        if (tar && tar->isEnabled()) {
            delete tar;
        } else {
            if (r->_tempFile) r->_tempFile.close();
        }
    }
}

/* ===================== /download ===================== */
inline void handleDownload(AsyncWebServerRequest *r) {

    if (!r->hasParam("path", true)) {
        r->send(400, "text/plain", "Missing path");
        return;
    }

    String path = r->getParam("path", true)->value();

    if (path == "/") {
        TarResponse *response = new TarResponse("/");
        response->addHeader("Content-Disposition",
                            "attachment; filename=\"fullFS.tar\"");
        r->send(response);
        return;
    }

    if (!LittleFS.exists(path)) {
        r->send(404, "text/plain", "Not found");
        return;
    }

    File f = LittleFS.open(path);
    bool isDir = f.isDirectory();
    f.close();

    if (!isDir) {
        r->send(LittleFS, path, "application/octet-stream", true);
        return;
    }

    String base = path.substring(path.lastIndexOf('/') + 1);
    if (base == "") base = "root";

    TarResponse *response = new TarResponse(path);
    response->addHeader("Content-Disposition",
                        "attachment; filename=\"" + base + ".tar\"");
    r->send(response);
}

/* ===================== /format ===================== */
inline void handleFormat(AsyncWebServerRequest *r) {
    if (!r->hasParam("confirm", true) ||
        r->getParam("confirm", true)->value() != "YES") {
        r->send(400, "application/json",
                "{\"ok\":false,\"error\":\"CONFIRM_REQUIRED\"}");
        return;
    }

    if (!LittleFS.format()) {
        r->send(500, "application/json",
                "{\"ok\":false,\"error\":\"FORMAT_FAILED\"}");
        return;
    }

    if (!LittleFS.begin(true)) {
        r->send(500, "application/json",
                "{\"ok\":false,\"error\":\"MOUNT_FAILED_AFTER_FORMAT\"}");
        return;
    }

    LittleFS.mkdir("/core");
    LittleFS.mkdir("/system");

    r->send(200, "application/json",
            "{\"ok\":true,\"msg\":\"FS formatted and core/system created\"}");
}

/* ===================== /mkdir ===================== */
inline void handleMkdir(AsyncWebServerRequest *r) {
    r->send(LittleFS.mkdir(r->getParam("path", true)->value()) ? 200 : 500);
}

/* ===================== /delete ===================== */
inline void handleDelete(AsyncWebServerRequest *r) {
    r->send(rm_rf(r->getParam("path", true)->value()) ? 200 : 500);
}

/* ===================== /rename ===================== */
inline void handleRename(AsyncWebServerRequest *r) {
    r->send(LittleFS.rename(
        r->getParam("old", true)->value(),
        r->getParam("new", true)->value()
    ) ? 200 : 500);
}

/* ===================== /core/admin/editor.html ===================== */
inline void handleEditor(AsyncWebServerRequest *r) {
    String path = "";
    if (r->hasParam("path", true)) path = r->getParam("path", true)->value();

    File f = LittleFS.open("/core/admin/editor.html", "r");
    if (!f) { r->send(404); return; }

    String h = f.readString();
    f.close();

    h.replace("<!--INSERT_PATH_HERE-->", path);
    r->send(200, "text/html", h);
}


/* ================= REKURZÍV TÖRLÉS ================= */
bool rm_rf(const String &path){
    File dir = LittleFS.open(path);
    if(!dir) return false;

    if(!dir.isDirectory()){
        dir.close();
        return LittleFS.remove(path);
    }

    std::vector<String> items;
    File f = dir.openNextFile();
    while(f){
        String name = String(f.name());
        if(!name.startsWith("/")) name = path + "/" + name;
        items.push_back(name);
        f.close();
        f = dir.openNextFile();
    }
    dir.close();

    for(const String &p : items){
        File x = LittleFS.open(p);
        if(x && x.isDirectory()){
            x.close();
            rm_rf(p);
        } else {
            if(x) x.close();
            LittleFS.remove(p);
        }
        yield();
    }

    return LittleFS.rmdir(path);
}
